﻿// Juicebox.jsx
// Copyright 2007-2009
// Written by Jeffrey Tranberry
// Photoshop for Geeks Version 3.0

// Updated for Juicebox v1.5.1 By Steven Speirs 2017

/////////////////////////////////////////////
// Juicebox v1.5.1
// https://www.juicebox.net/
/////////////////////////////////////////////

// enable double clicking from the
// Macintosh Finder or the Windows Explorer
#target photoshop

// Make Photoshop the frontmost application
// in case we double clicked the file
app.bringToFront();

///////////////////////////
//       SET-UP
///////////////////////////

var gVersion = "1.5.1";

// a global variable for the title of the dialog
// this string will also be used for the preferences file I write to disk
// Photoshop Install Directory/Presets/Image Processor/Image Processor.xml for example

var gScriptName = "Juicebox";

// remember the dialog modes
var saveDialogMode = app.displayDialogs;
app.displayDialogs = DialogModes.NO;

try {

	// make sure they are running Photoshop CS2
	CheckVersion();

}

// Lot's of things can go wrong, Give a generic alert and see if they want the details
catch(e) {
	if ( confirm("Sorry, something major happened and I can't continue! Would you like to see more info?" ) ) {
		alert(e);
	}
}

// Save the current preferences
	var startRulerUnits = app.preferences.rulerUnits;
	var startTypeUnits = app.preferences.typeUnits;
	var startDisplayDialogs = app.displayDialogs;

// Set Photoshop to use pixels and display no dialogs
	app.preferences.rulerUnits = Units.PIXELS;
	app.preferences.typeUnits = TypeUnits.PIXELS;
	app.displayDialogs = DialogModes.NO;

// Set the script location

	var scriptLocation = findScript() + "0";

// A list of file extensions to skip, keep them lower case

	var gFilesToSkip = Array( "db", "xmp", "thm", "txt", "doc", "md0", "tb0", "adobebridgedb", "adobebridgedbt", "bc", "bct", "js", "html", "mov", "xml", "swf" );

	var strButtonBrowse = localize("$$$/JavaScripts/ExportLayersToFiles/Browse=Browse...");
	var strAlertSpecifyDestination = localize("$$$/JavaScripts/ExportLayersToFiles/SpecifyDestination=Please specify an output folder.");
	var strAlertDestinationNotExist = localize("$$$/JavaScripts/ExportLayersToFiles/DestionationDoesNotExist=Output folder does not exist.");
	var strAlertSpecifyInputFolder = localize("$$$/JavaScripts/ExportLayersToFiles/SpecifyInputFolder=Please specify an input folder.");
	var strAlertInputFolderNotExist = localize("$$$/JavaScripts/ExportLayersToFiles/InputFolderDoesNotExist=Input folder does not exist.");

	var exportInfo = new Object();

	initExportInfo(exportInfo);

	var inputBrowseClicked = false;
	var outputBrowseClicked = false;

// define the dialog
// [left, top, right, bottom]
function createDialog(){

	// Create an empty dialog window near the upper left of the screen
	var dlg = new Window('dialog', 'Juicebox');
	dlg.frameLocation = [100, 100];
	dlg.orientation = "row";

	dlg.msgPnl = dlg.add('panel', undefined, 'Lite Options');
	dlg.msgPnl.orientation = "column";
	dlg.msgPnl.alignChildren = "left";

	dlg.msgPnl.maxImageWidth = dlg.msgPnl.add('group');
	dlg.msgPnl.maxImageWidth.orientation = "row";
	dlg.msgPnl.maxImageWidth.st = dlg.msgPnl.maxImageWidth.add('statictext', undefined, 'Maximum Image Width, px:');
	dlg.msgPnl.maxImageWidth.et = dlg.msgPnl.maxImageWidth.add('edittext', undefined, '1024');
	dlg.msgPnl.maxImageWidth.et.preferredSize = [60,20];
	dlg.msgPnl.maxImageWidth.helpTip = "The maximum image width, in pixels, of the main images.";

	dlg.msgPnl.maxImageHeight = dlg.msgPnl.add('group');
	dlg.msgPnl.maxImageHeight.orientation = "row";
	dlg.msgPnl.maxImageHeight.st = dlg.msgPnl.maxImageHeight.add('statictext', undefined, 'Maximum Image Height, px:');
	dlg.msgPnl.maxImageHeight.et = dlg.msgPnl.maxImageHeight.add('edittext', undefined, '768');
	dlg.msgPnl.maxImageHeight.et.preferredSize = [60,20];
	dlg.msgPnl.maxImageHeight.helpTip = "The maximum image height, in pixels, of the main images.";

	dlg.msgPnl.galleryTitle = dlg.msgPnl.add('group');
	dlg.msgPnl.galleryTitle.orientation = "row";
	dlg.msgPnl.galleryTitle.st = dlg.msgPnl.galleryTitle.add('statictext', undefined, 'Gallery Title:');
	dlg.msgPnl.galleryTitle.et = dlg.msgPnl.galleryTitle.add('edittext', undefined, 'Juicebox Gallery');
	dlg.msgPnl.galleryTitle.et.preferredSize = [175,20];
	dlg.msgPnl.galleryTitle.helpTip = "Enter a Gallery Title.";

	dlg.msgPnl.backgroundColor = dlg.msgPnl.add('group');
	dlg.msgPnl.backgroundColor.orientation = "row";
	dlg.msgPnl.backgroundColor.st = dlg.msgPnl.backgroundColor.add('statictext', undefined, 'Background Color #:');
	dlg.msgPnl.backgroundColor.et = dlg.msgPnl.backgroundColor.add('edittext', undefined, '222222');
	dlg.msgPnl.backgroundColor.et.preferredSize = [60,20];
	dlg.msgPnl.backgroundColor.helpTip = "Choose a hexadecimal value for the background color.";

	dlg.msgPnl.backgroundOpacity = dlg.msgPnl.add('group');
	dlg.msgPnl.backgroundOpacity.orientation = "row";
	dlg.msgPnl.backgroundOpacity.st = dlg.msgPnl.backgroundOpacity.add('statictext', undefined, 'Background Opacity:');
	dlg.msgPnl.backgroundOpacity.et = dlg.msgPnl.backgroundOpacity.add('edittext', undefined, '1');
	dlg.msgPnl.backgroundOpacity.et.preferredSize = [60,20];
	dlg.msgPnl.backgroundOpacity.helpTip = "Choose the opacity for the gallery background (between 0 and 1).";

	dlg.msgPnl.textColor = dlg.msgPnl.add('group');
	dlg.msgPnl.textColor.orientation = "row";
	dlg.msgPnl.textColor.st = dlg.msgPnl.textColor.add('statictext', undefined, 'Text Color #:');
	dlg.msgPnl.textColor.et = dlg.msgPnl.textColor.add('edittext', undefined, 'ffffff');
	dlg.msgPnl.textColor.et.preferredSize = [60,20];
	dlg.msgPnl.textColor.helpTip = "Choose a hexadecimal value for the text color.";

	dlg.msgPnl.textOpacity = dlg.msgPnl.add('group');
	dlg.msgPnl.textOpacity.orientation = "row";
	dlg.msgPnl.textOpacity.st = dlg.msgPnl.textOpacity.add('statictext', undefined, 'Text Opacity:');
	dlg.msgPnl.textOpacity.et = dlg.msgPnl.textOpacity.add('edittext', undefined, '1');
	dlg.msgPnl.textOpacity.et.preferredSize = [60,20];
	dlg.msgPnl.textOpacity.helpTip = "Choose the opacity for the text (between 0 and 1).";

	dlg.msgPnl.thumbColor = dlg.msgPnl.add('group');
	dlg.msgPnl.thumbColor.orientation = "row";
	dlg.msgPnl.thumbColor.st = dlg.msgPnl.thumbColor.add('statictext', undefined, 'Thumb Frame Color #:');
	dlg.msgPnl.thumbColor.et = dlg.msgPnl.thumbColor.add('edittext', undefined, 'ffffff');
	dlg.msgPnl.thumbColor.et.preferredSize = [60,20];
	dlg.msgPnl.thumbColor.helpTip = "Choose a hexadecimal value for the thumbnail frames color.";

	dlg.msgPnl.thumbOpacity = dlg.msgPnl.add('group');
	dlg.msgPnl.thumbOpacity.orientation = "row";
	dlg.msgPnl.thumbOpacity.st = dlg.msgPnl.thumbOpacity.add('statictext', undefined, 'Thumb Frame Opacity:');
	dlg.msgPnl.thumbOpacity.et = dlg.msgPnl.thumbOpacity.add('edittext', undefined, '0.5');
	dlg.msgPnl.thumbOpacity.et.preferredSize = [60,20];
	dlg.msgPnl.thumbOpacity.helpTip = "Choose the opacity for the thumbnail frames (between 0 and 1).";

	dlg.msgPnl.showOpenButton = dlg.msgPnl.add('group');
	dlg.msgPnl.showOpenButton.orientation = "row";
	dlg.msgPnl.showOpenButton.st = dlg.msgPnl.showOpenButton.add('checkbox', undefined, 'Show Open Image Button');
	dlg.msgPnl.showOpenButton.st.value = true;
	dlg.msgPnl.showOpenButton.helpTip = "Whether to show 'Open Image' button.";

	dlg.msgPnl.showExpandButton = dlg.msgPnl.add('group');
	dlg.msgPnl.showExpandButton.orientation = "row";
	dlg.msgPnl.showExpandButton.st = dlg.msgPnl.showExpandButton.add('checkbox', undefined, 'Show Expand Button');
	dlg.msgPnl.showExpandButton.st.value = true;
	dlg.msgPnl.showExpandButton.helpTip = "Whether to show 'Expand' button. Clicking this button expands the gallery to fill the browser window.";

	dlg.msgPnl.showThumbsButton = dlg.msgPnl.add('group');
	dlg.msgPnl.showThumbsButton.orientation = "row";
	dlg.msgPnl.showThumbsButton.st = dlg.msgPnl.showThumbsButton.add('checkbox', undefined, 'Show Thumbs Button');
	dlg.msgPnl.showThumbsButton.st.value = true;
	dlg.msgPnl.showThumbsButton.helpTip = "Whether to show the 'Toggle Thumbnails' button in Large Screen Mode.";

	dlg.msgPnl.useThumbDots = dlg.msgPnl.add('group');
	dlg.msgPnl.useThumbDots.orientation = "row";
	dlg.msgPnl.useThumbDots.st = dlg.msgPnl.useThumbDots.add('checkbox', undefined, 'Use Thumb Dots');
	dlg.msgPnl.useThumbDots.st.value = false;
	dlg.msgPnl.useThumbDots.helpTip = "Replace the thumbnail images with small dots.";

	dlg.msgPnl.useFullscreenExpand = dlg.msgPnl.add('group');
	dlg.msgPnl.useFullscreenExpand.orientation = "row";
	dlg.msgPnl.useFullscreenExpand.st = dlg.msgPnl.useFullscreenExpand.add('checkbox', undefined, 'Use Fullscreen Expand');
	dlg.msgPnl.useFullscreenExpand.st.value = false;
	dlg.msgPnl.useFullscreenExpand.helpTip = "Triggers fullscreen mode when clicking the 'expand' button for supported browsers (Firefox, Safari and Chrome).";

	dlg.wrapPnl = dlg.add('panel', undefined, undefined, {borderStyle: 'none'});
	dlg.wrapPnl.orientation = "column";

	// Add a panel to hold title and 'message text' strings
	dlg.msgPn2 = dlg.wrapPnl.add('panel', undefined, 'Input Folder');
	dlg.msgPn2.orientation = "column";
	dlg.msgPn2.alignChildren = "left";

	dlg.msgPn2.InputFolder = dlg.msgPn2.add('group');
	dlg.msgPn2.InputFolder.orientation = "row";
	dlg.msgPn2.etInputFolder = dlg.msgPn2.add("edittext", undefined, exportInfo.destination.toString());
	dlg.msgPn2.etInputFolder.preferredSize.width = 275;
	dlg.msgPn2.etInputFolder.helpTip = "Choose a folder of images to process.";

	dlg.msgPn2.btnBrowse = dlg.msgPn2.add("button", undefined, strButtonBrowse);
	dlg.msgPn2.btnBrowse.helpTip = "Choose a folder of images to process.";
	dlg.msgPn2.btnBrowse.onClick = function() {
		inputBrowseClicked = true;
		var defaultFolder = dlg.msgPn2.etInputFolder.text;
		var testFolder = new Folder(dlg.msgPn2.etInputFolder.text);
		if (!testFolder.exists) {
			defaultFolder = "~";
		}
		dlg.selInputFolder = Folder.selectDialog(dlg.msgPn2.etInputFolder.text, defaultFolder);
		if ( dlg.selInputFolder != null ) {
	        dlg.msgPn2.etInputFolder.text = dlg.selInputFolder.fsName;

	    }
	}

	// Add a panel to hold title and 'message text' strings
	dlg.msgPn3 = dlg.wrapPnl.add('panel', undefined, 'Output Folder');
	dlg.msgPn3.orientation = "column";
	dlg.msgPn3.alignChildren = "left";

	dlg.msgPn3.Destination = dlg.msgPn3.add('group');
	dlg.msgPn3.Destination.orientation = "row";
	dlg.msgPn3.etDestination = dlg.msgPn3.add("edittext", undefined, exportInfo.destination.toString());
	dlg.msgPn3.etDestination.preferredSize.width = 275;
	dlg.msgPn3.etDestination.helpTip = "Choose a folder to export your web gallery to.";

	dlg.msgPn3.btnBrowse = dlg.msgPn3.add("button", undefined, strButtonBrowse);
	dlg.msgPn3.btnBrowse.helpTip = "Choose a folder to export your web gallery to.";
	dlg.msgPn3.btnBrowse.onClick = function() {
		outputBrowseClicked = true;
		var defaultFolder = dlg.msgPn3.etDestination.text;
		var testFolder = new Folder(dlg.msgPn3.etDestination.text);
		if (!testFolder.exists) {
			defaultFolder = "~";
		}
		dlg.selOutputFolder = Folder.selectDialog(dlg.msgPn3.etDestination.text, defaultFolder);
		if ( dlg.selOutputFolder != null ) {
	        dlg.msgPn3.etDestination.text = dlg.selOutputFolder.fsName;
	    }
	}

	// Add a panel with buttons
	dlg.buttonPanel = dlg.wrapPnl.add('panel', undefined, undefined, {borderStyle: 'none'});
	dlg.buttonPanel.orientation = "row";
	dlg.buttonPanel.runBtn = dlg.buttonPanel.add ('button', undefined, 'Build Web Gallery');
	dlg.buttonPanel.cancelBtn = dlg.buttonPanel.add ('button', undefined, 'Cancel');

	return dlg;
	}

	var params = new Array();
	params['MaxImageWidth'] = "1024";
	params['MaxImageHeight'] = "768";
	params['GalleryTitle'] = "Juicebox Gallery";
	params['BackgroundColor'] = "222222";
	params['BackgroundOpacity'] = "1";
	params['TextColor'] = "ffffff";
	params['TextOpacity'] = "1";
	params['ThumbColor'] = "ffffff";
	params['ThumbOpacity'] = "0.5";
	params['ShowOpenButton'] = true;
	params['ShowExpandButton'] = true;
	params['ShowThumbsButton'] = true;
	params['UseThumbDots'] = false;
	params['UseFullscreenExpand'] = false;
	params['InputFolder'] = "";
	params['OutputFolder'] = "";

	LoadParamsFromDisk( GetDefaultParamsFile(), params );

	function initializeDialog (Juicebox){
	with(Juicebox) {

	msgPnl.maxImageWidth.et.text = params['MaxImageWidth'];
	msgPnl.maxImageHeight.et.text = params['MaxImageHeight'];
	msgPnl.galleryTitle.et.text = params['GalleryTitle'];
	msgPnl.backgroundColor.et.text = params['BackgroundColor'];
	msgPnl.backgroundOpacity.et.text = params['BackgroundOpacity'];
	msgPnl.textColor.et.text = params['TextColor'];
	msgPnl.textOpacity.et.text = params['TextOpacity'];
	msgPnl.thumbColor.et.text = params['ThumbColor'];
	msgPnl.thumbOpacity.et.text = params['ThumbOpacity'];
	msgPnl.showOpenButton.st.value = params['ShowOpenButton'];
	msgPnl.showExpandButton.st.value = params['ShowExpandButton'];
	msgPnl.showThumbsButton.st.value = params['ShowThumbsButton'];
	msgPnl.useThumbDots.st.value = params['UseThumbDots'];
	msgPnl.useFullscreenExpand.st.value = params['UseFullscreenExpand'];
	msgPn2.etInputFolder.text = params['InputFolder'];
	msgPn3.etDestination.text = params['OutputFolder'];

	// Run the web photo gallery
	// check for valid settings
	buttonPanel.runBtn.onClick = function() {

	// check if the input folder setting is valid
			var inptfld = Juicebox.msgPn2.etInputFolder.text;
			if (inptfld.length == 0) {
				alert(strAlertSpecifyInputFolder);
				return;
			}
			var testFolder = new Folder(inptfld);
			if (!testFolder.exists) {
				alert(strAlertInputFolderNotExist);
				return;
			}

			// check if the output folder setting is valid
			var destination = Juicebox.msgPn3.etDestination.text;
			if (destination.length == 0) {
				alert(strAlertSpecifyDestination);
				return;
			}
			var testFolder = new Folder(destination);
			if (!testFolder.exists) {
				alert(strAlertDestinationNotExist);
				return;
			}

			// check if the input folder and the output folder are the same
			if (Juicebox.msgPn3.etDestination.text == Juicebox.msgPn2.etInputFolder.text) {
				var result = confirm("Are you sure you want your output folder to be the same as your input folder?");
				if (result) {
				} else {
					return;
				}
    		}

    		// check if the Maximum Image Width setting is valid
			var imageSpaceStr = isNaN(Juicebox.msgPnl.maxImageWidth.et.text);
			if (imageSpaceStr == true) {
        		alert("Please enter a valid number 300-5000 for 'Maximum Image Width'.");
        		return;
        	}
    		if (!(Juicebox.msgPnl.maxImageWidth.et.text >= 300 && Juicebox.msgPnl.maxImageWidth.et.text <= 5000)) {
				alert("You must enter a number between 300-5000 for 'Maximum Image Width'.");
				return;
    		}

     		// check if the Maximum Image Height setting is valid
			var imageSpaceStr = isNaN(Juicebox.msgPnl.maxImageHeight.et.text);
			if (imageSpaceStr == true) {
        		alert("Please enter a valid number 300-5000 for 'Maximum Image Height'.");
        		return;
        	}
    		if (!(Juicebox.msgPnl.maxImageHeight.et.text >= 300 && Juicebox.msgPnl.maxImageWidth.et.text <= 5000)) {
				alert("You must enter a number between 300-5000 for 'Maximum Image Height'.");
				return;
    		}

			// check if the Background Opacity setting is valid
			var backgroundOpacityStr = isNaN(Juicebox.msgPnl.backgroundOpacity.et.text);
			if (backgroundOpacityStr == true) {
        		alert("Please enter a valid number 0-1 for 'Background Opacity'.");
        		return;
        	}
    		if (!(Juicebox.msgPnl.backgroundOpacity.et.text >= 0 && Juicebox.msgPnl.backgroundOpacity.et.text <= 1)) {
				alert("You must enter a number between 0-1 for 'Background Opacity'.");
				return;
    		}

	close( 1 );
	}
	buttonPanel.cancelBtn.onClick = function() { close( 2 ); }
	}
	}

	function runDialog(Juicebox){

	// Warn the user if they have an open document and exit the script with return
	if (documents.length > 0){
		alert ("This script requires that there are no open documents to run.");
	return;
	}

	return Juicebox.show()
	}

	var Juicebox = createDialog()
	initializeDialog(Juicebox)

//=====================Start=====================================================

	if (runDialog(Juicebox) == 1){
	// transfer values from the dialog to my internal params
	params['MaxImageWidth'] = Juicebox.msgPnl.maxImageWidth.et.text;
	params['MaxImageHeight'] = Juicebox.msgPnl.maxImageHeight.et.text;
	params['GalleryTitle'] = Juicebox.msgPnl.galleryTitle.et.text;
	params['BackgroundColor'] = Juicebox.msgPnl.backgroundColor.et.text;
	params['BackgroundOpacity'] = Juicebox.msgPnl.backgroundOpacity.et.text;
	params['TextColor'] = Juicebox.msgPnl.textColor.et.text;
	params['TextOpacity'] = Juicebox.msgPnl.textOpacity.et.text;
	params['ThumbColor'] = Juicebox.msgPnl.thumbColor.et.text;
	params['ThumbOpacity'] = Juicebox.msgPnl.thumbOpacity.et.text;
	params['ShowOpenButton'] = Juicebox.msgPnl.showOpenButton.st.value;
	params['ShowExpandButton'] = Juicebox.msgPnl.showExpandButton.st.value;
	params['ShowThumbsButton'] = Juicebox.msgPnl.showThumbsButton.st.value;
	params['UseThumbDots'] = Juicebox.msgPnl.useThumbDots.st.value;
	params['UseFullscreenExpand'] = Juicebox.msgPnl.useFullscreenExpand.st.value;
	params['InputFolder'] = Juicebox.msgPn2.etInputFolder.text;
	params['OutputFolder'] = Juicebox.msgPn3.etDestination.text;

	// Save the params from the above
	SaveParamsToDisk( GetDefaultParamsFile(), params );

			// Gets the input folder from the UI
			var inputFolder = Juicebox.msgPn2.etInputFolder.text;

			// Gets the output folder from the UI
			var outputFolder = Juicebox.msgPn3.etDestination.text;

			// Gets the Maximum Image Width from the UI
			var maxImageWidth = Juicebox.msgPnl.maxImageWidth.et.text;

			// Gets the Maximum Image Height from the UI
			var maxImageHeight = Juicebox.msgPnl.maxImageHeight.et.text;

			// Determine line feed and then save out text files.  Note that "Desktop" is localized
			// on Win, when so we don't reference the desktop if it can not be found.

				if (File.fs == "Windows") {
					fileLineFeed = "windows";
					var desktopFolder = Folder("~/" + outputFolder);
					if (desktopFolder.exists != 1) {
						var imageFolder = new Folder(outputFolder + "/images");
						var thumbsFolder = new Folder(outputFolder + "/thumbs");
						var jbcoreFolder = new Folder(outputFolder + "/jbcore");
						var classicFolder = new Folder(outputFolder + "/jbcore/classic");
						var classicFontsFolder = new Folder(outputFolder + "/jbcore/classic/fonts");
						var classicImgFolder = new Folder(outputFolder + "/jbcore/classic/img");
					} else {
						var imageFolder = new Folder("~/" + outputFolder + "/images");
						var thumbsFolder = new Folder("~/" + outputFolder + "/thumbs");
						var jbcoreFolder = new Folder("~/" + outputFolder + "/jbcore");
						var classicFolder = new Folder("~/" + outputFolder + "/jbcore/classic");
						var classicFontsFolder = new Folder("~/" + outputFolder + "/jbcore/classic/fonts");
						var classicImgFolder = new Folder("~/" + outputFolder + "/jbcore/classic/img");
					}
				} else {
					fileLineFeed = "macintosh";
					var imageFolder = new Folder(outputFolder + "/images");
					var thumbsFolder = new Folder(outputFolder + "/thumbs");
					var jbcoreFolder = new Folder(outputFolder + "/jbcore");
					var classicFolder = new Folder(outputFolder + "/jbcore/classic");
					var classicFontsFolder = new Folder(outputFolder + "/jbcore/classic/fonts");
					var classicImgFolder = new Folder(outputFolder + "/jbcore/classic/img");
				}
				imageFolder.create();
				thumbsFolder.create();
				jbcoreFolder.create();
				classicFolder.create();
				classicFontsFolder.create();
				classicImgFolder.create();

			// Generate HTML File

			var HTMLPath = outputFolder + "/" + "index.html";
			var HTMLFile = File(HTMLPath);
			HTMLFile.remove();

			// Generate XML File

    		var XMLPath = outputFolder + "/" + "config.xml";
			var XMLFile = File(XMLPath);
			XMLFile.remove();

			// Copy ReadMe File from the Presets/Scripts folder to the specified output folder

			var file1Path = scriptLocation.slice(0, -13) + "readme.html";
			var file1File = File(file1Path);
			file1File.copy(outputFolder + "/readme.html");

			// Copy JBCORE Folder from the Presets/Scripts folder to the specified output folder

			var file2Path = scriptLocation.slice(0, -13) + "jbcore/full.html";
			var file2File = File(file2Path);
			file2File.copy(outputFolder + "/jbcore/full.html");

			var file3Path = scriptLocation.slice(0, -13) + "jbcore/~juicebox.js";
			var file3File = File(file3Path);
			file3File.copy(outputFolder + "/jbcore/juicebox.js");

			var file4Path = scriptLocation.slice(0, -13) + "jbcore/juicebox.js";
			var file4File = File(file4Path);
			file4File.copy(outputFolder + "/jbcore/juicebox.js");

			var file5Path = scriptLocation.slice(0, -13) + "jbcore/juicebox.php";
			var file5File = File(file5Path);
			file5File.copy(outputFolder + "/jbcore/juicebox.php");

			var file6Path = scriptLocation.slice(0, -13) + "jbcore/pswd.php";
			var file6File = File(file6Path);
			file6File.copy(outputFolder + "/jbcore/pswd.php");

			var file7Path = scriptLocation.slice(0, -13) + "jbcore/classic/theme.css";
			var file7File = File(file7Path);
			file7File.copy(outputFolder + "/jbcore/classic/theme.css");

			var file8Path = scriptLocation.slice(0, -13) + "jbcore/classic/fonts/juicebox.eot";
			var file8File = File(file8Path);
			file8File.copy(outputFolder + "/jbcore/classic/fonts/juicebox.eot");

			var file9Path = scriptLocation.slice(0, -13) + "jbcore/classic/fonts/juicebox.svg";
			var file9File = File(file9Path);
			file9File.copy(outputFolder + "/jbcore/classic/fonts/juicebox.svg");

			var file10Path = scriptLocation.slice(0, -13) + "jbcore/classic/fonts/juicebox.ttf";
			var file10File = File(file10Path);
			file10File.copy(outputFolder + "/jbcore/classic/fonts/juicebox.ttf");

			var file11Path = scriptLocation.slice(0, -13) + "jbcore/classic/fonts/juicebox.woff";
			var file11File = File(file11Path);
			file11File.copy(outputFolder + "/jbcore/classic/fonts/juicebox.woff");

			var file12Path = scriptLocation.slice(0, -13) + "jbcore/classic/img/jb001.png";
			var file12File = File(file12Path);
			file12File.copy(outputFolder + "/jbcore/classic/img/jb001.png");

			var file13Path = scriptLocation.slice(0, -13) + "jbcore/classic/img/spinner.gif";
			var file13File = File(file13Path);
			file13File.copy(outputFolder + "/jbcore/classic/img/spinner.gif");

			// Write out XML Header to the file created in the set-up

			var textRGBA=getRGBA(Juicebox.msgPnl.textColor.et.text, Juicebox.msgPnl.textOpacity.et.text);
			var thumbRGBA=getRGBA(Juicebox.msgPnl.thumbColor.et.text, Juicebox.msgPnl.thumbOpacity.et.text);

			writeXMLHeader(Juicebox.msgPnl.galleryTitle.et.text, textRGBA, thumbRGBA, Juicebox.msgPnl.showOpenButton.st.value, Juicebox.msgPnl.showExpandButton.st.value, Juicebox.msgPnl.showThumbsButton.st.value, Juicebox.msgPnl.useThumbDots.st.value, Juicebox.msgPnl.useFullscreenExpand.st.value);

			// Open and process a folder of Images
			if(!inputBrowseClicked || !outputBrowseClicked){
				var manInputFolder = Folder(inputFolder);
				var manOutputFolder = Folder(outputFolder);
				OpenFolder(manInputFolder, manOutputFolder, maxImageWidth, maxImageHeight);
			}else{
				OpenFolder(Juicebox.selInputFolder, Juicebox.selOutputFolder, maxImageWidth, maxImageHeight);
			}
			// Write out XML Footer to the file created in the set-up

			writeXMLFooter();

			// Write out all the HTML to the file created in the set-up

			var backgroundRGBA=getRGBA(Juicebox.msgPnl.backgroundColor.et.text, Juicebox.msgPnl.backgroundOpacity.et.text);

			writeHTMLDoc(Juicebox.msgPnl.galleryTitle.et.text, backgroundRGBA);

			// launch the web gallery in a browser

			File(outputFolder + "/index.html").execute();

	// Return the app preferences
	app.preferences.rulerUnits = startRulerUnits;
	app.preferences.typeUnits = startTypeUnits;
	app.displayDialogs = saveDialogMode;

} // the end

///////////////////////
// Helper Functions
///////////////////////

///////////////////////////////////////////////////////////////////////////////
// Function: initExportInfo
// Usage: create our default parameters
// Input: a new Object
// Return: a new object with params set to default
///////////////////////////////////////////////////////////////////////////////
function initExportInfo(exportInfo) {
    exportInfo.destination = new String("");
    exportInfo.fileNamePrefix = new String("untitled_");
    exportInfo.visibleOnly = false;
    exportInfo.icc = true;
    exportInfo.jpegQuality = 8;
    exportInfo.psdMaxComp = true;
    exportInfo.tiffCompression = TIFFEncoding.NONE;
    exportInfo.tiffJpegQuality = 8;
    exportInfo.pdfEncoding = PDFEncoding.JPEG;
    exportInfo.pdfJpegQuality = 8;
    exportInfo.targaDepth = TargaBitsPerPixels.TWENTYFOUR;
    exportInfo.bmpDepth = BMPDepthType.TWENTYFOUR;

    try {
        exportInfo.destination = Folder(app.activeDocument.fullName.parent).fsName; // destination folder
        var tmp = app.activeDocument.fullName.name;
        exportInfo.fileNamePrefix = decodeURI(tmp.substring(0, tmp.indexOf("."))); // filename body part
    } catch(someError) {
        exportInfo.destination = new String("");
    }
}

// Find the location where this script resides

function findScript() {
	var where = "";
	try {
		FORCEERROR = FORCERRROR;
	}
	catch(err) {
		where = File(err.fileName);
	}
	return where;
}

////////////////////////////
// File Writing Functions
////////////////////////////

// Write out image name XML

    function writeXMLDocRef() {
        writeXML(
			'	<image imageURL="images/' + app.activeDocument.name + '" thumbURL="thumbs/' + app.activeDocument.name + '" linkURL="" linkTarget="" >\n' +
			'		<title><![CDATA[' + escapeCDATA(stripControlCharacters(lineBreak(app.activeDocument.name))) + ']]></title>\n' +
			'		<caption><![CDATA[]]></caption>\n' +
			'	</image>\n'
		);
    }

// Write out XML header

    function writeXMLHeader(galleryTitle, textColor, thumbFrameColor, showOpenButton, showExpandButton, showThumbsButton, useThumbDots, useFullscreenExpand) {
		writeXML(
			'<?xml version="1.0" encoding="UTF-8"?>\n' +
			'<juiceboxgallery\n' +
			'	useFlickr="false"\n' +
			'	galleryTitle="' + escapeXML(lineBreak(galleryTitle)) + '"\n' +
			'	textColor="' + textColor + '"\n' +
			'	thumbFrameColor="' + thumbFrameColor + '"\n' +
			'	showOpenButton="' + showOpenButton + '"\n' +
			'	showExpandButton="' + showExpandButton + '"\n' +
			'	showThumbsButton="' + showThumbsButton + '"\n' +
			'	useThumbDots="' + useThumbDots + '"\n' +
			'	useFullscreenExpand="' + useFullscreenExpand + '"\n' +
			'>\n'
		);
    }

// Write out XML footer

    function writeXMLFooter() {
		writeXML('</juiceboxgallery>\n');
    }

// Write XML file

    function writeXML(log) {
        try {
            if(XMLFile.exists) {
                XMLFile.open ("e");
                XMLFile.seek (0,2);      // Move to EOF
            } else {
            XMLFile.open ("w");          // Add unicode marker if we change to XML file format for this log file
        }
        XMLFile.encoding = "UTF8"; // set UTF8
        XMLFile.write(log);
        XMLFile.close();
        } catch (e) {
            alert(e);
        } finally {
        }
        return;
    }

// Write index HTML file

    function writeHTML(log) {
        try {
            if(HTMLFile.exists) {
                HTMLFile.open ("e");
                HTMLFile.seek (0,2);      // Move to EOF
            } else {
            HTMLFile.open ("w");          // Add unicode marker if we change to XML file format for this log file
        }
        HTMLFile.encoding = "UTF8"; // set UTF8
        HTMLFile.write(log);
        HTMLFile.close();
        } catch (e) {
            alert(e);
        } finally {
        }
        return;
    }

/*/ Write upgrade HTML file

    function writeUGHTML(log) {
        try {
            if(UGHTMLFile.exists) {
                UGHTMLFile.open ("e");
                UGHTMLFile.seek (0,2);      // Move to EOF
            } else {
            UGHTMLFile.open ("w");          // Add unicode marker if we change to XML file format for this log file
        }
        UGHTMLFile.encoding = "UTF8"; // set UTF8
        UGHTMLFile.write(log);
        UGHTMLFile.close();
        } catch (e) {
            alert(e);
        } finally {
        }
        return;
    }
*/

// Function for returning current date and time

    function getDateTime() {
        var date = new Date();
        var dateTime = "";
        if ((date.getMonth() + 1) < 10) {
            dateTime += "0" + (date.getMonth() + 1) + "/";
        } else {
            dateTime += (date.getMonth() + 1) + "/";
        }
        if (date.getDate() < 10) {
            dateTime += "0" + date.getDate() + "/";
        } else {
            dateTime += date.getDate() + "/";
        }
        dateTime += date.getFullYear() + ", ";
        if (date.getHours() < 10) {
            dateTime += "0" + date.getHours() + ":";
        } else {
            dateTime += date.getHours() + ":";
        }
        if (date.getMinutes() < 10) {
            dateTime += "0" + date.getMinutes() + ":";
        } else {
            dateTime += date.getMinutes() + ":";
        }
        if (date.getSeconds() < 10) {
            dateTime += "0" + date.getSeconds();
        } else {
            dateTime += date.getSeconds();
        }
        return dateTime;
    }

// resetPrefs function for resetting the preferences

	function resetPrefs() {
		preferences.rulerUnits = startRulerUnits;
		preferences.typeUnits = startTypeUnits;
		displayDialogs = startDisplayDialogs;
	}

////////////////////////////////////
// File Opening/Closing Functions
////////////////////////////////////

// Given the a Folder of files, open them

function OpenFolder(inputFolder, outputFolder, maxImageWidth, maxImageHeight) {
        var filesOpened = 0;
        var fileList = inputFolder.getFiles();
        for ( var i = 0; i < fileList.length; i++ ) {
        		   // Make sure all the files in the folder are compatible with PS
                if ( fileList[i] instanceof File && ! fileList[i].hidden && ! IsFileOneOfThese( fileList[i], gFilesToSkip )) {
					open( fileList[i] );
					filesOpened++;
					var docRef = activeDocument;

					if (docRef.height > maxImageHeight || docRef.width > maxImageWidth)
					{
						if (docRef.height / maxImageHeight > docRef.width / maxImageWidth)
						{
							resizePortrait(maxImageHeight);
						}
						else
						{
							resizeLandscape(maxImageWidth);
						}
					}

					resolveColorModes();

					app.activeDocument.convertProfile("sRGB IEC61966-2.1", Intent.RELATIVECOLORIMETRIC);

					app.activeDocument.flatten();

					saveJPG(outputFolder + "/images/");

					writeXMLDocRef();

					if (docRef.height > docRef.width)
					{
						//resizePortrait(85); // use this for max width
						resizeLandscape(85); // use this for min width (only one or the other)
					}
					else
					{
						//resizeLandscape(85); // use this for max width
						resizePortrait(85); // use this for min width (only one or the other)
					}

					saveJPG(outputFolder + "/thumbs/");

                 fileClose(SaveOptions.DONOTSAVECHANGES);

                }
        }
        return filesOpened;
}

// given a file name and a list of extensions
// determine if this file is in the list of extensions

function IsFileOneOfThese( inFileName, inArrayOfFileExtensions ) {
	var lastDot = inFileName.toString().lastIndexOf( "." );
	if ( lastDot == -1 ) {
		return false;
	}
	var strLength = inFileName.toString().length;
	var extension = inFileName.toString().substr( lastDot + 1, strLength - lastDot );
	extension = extension.toLowerCase();
	for (var i = 0; i < inArrayOfFileExtensions.length; i++ ) {
		if ( extension == inArrayOfFileExtensions[i] ) {
			return true;
		}
	}
	return false;
}

// fileClose function for closing current document

function fileClose(options) {

		app.activeDocument.close(options);
}

////////////////////////////////
// Image Processing Functions
////////////////////////////////

// resize landscape

function resizeLandscape(maxImageWidth) {
var id3 = charIDToTypeID( "ImgS" );
    var desc2 = new ActionDescriptor();
    var id4 = charIDToTypeID( "Wdth" );
    var id5 = charIDToTypeID( "#Pxl" );
    desc2.putUnitDouble( id4, id5, maxImageWidth );
    var id6 = charIDToTypeID( "Rslt" );
    var id7 = charIDToTypeID( "#Rsl" );
    desc2.putUnitDouble( id6, id7, 72.000000 );
    var id8 = stringIDToTypeID( "scaleStyles" );
    desc2.putBoolean( id8, true );
    var id9 = charIDToTypeID( "CnsP" );
    desc2.putBoolean( id9, true );
    var id10 = charIDToTypeID( "Intr" );
    var id11 = charIDToTypeID( "Intp" );
    var id12 = charIDToTypeID( "Bcbc" );
    desc2.putEnumerated( id10, id11, id12 );
executeAction( id3, desc2, DialogModes.NO );
}

// resize Portrait

function resizePortrait(maxImageHeight) {
var id15 = charIDToTypeID( "ImgS" );
    var desc4 = new ActionDescriptor();
    var id16 = charIDToTypeID( "Hght" );
    var id17 = charIDToTypeID( "#Pxl" );
    desc4.putUnitDouble( id16, id17, maxImageHeight );
    var id18 = charIDToTypeID( "Rslt" );
    var id19 = charIDToTypeID( "#Rsl" );
    desc4.putUnitDouble( id18, id19, 72.000000 );
    var id20 = stringIDToTypeID( "scaleStyles" );
    desc4.putBoolean( id20, true );
    var id21 = charIDToTypeID( "CnsP" );
    desc4.putBoolean( id21, true );
    var id22 = charIDToTypeID( "Intr" );
    var id23 = charIDToTypeID( "Intp" );
    var id24 = charIDToTypeID( "Bcbc" );
    desc4.putEnumerated( id22, id23, id24 );
executeAction( id15, desc4, DialogModes.NO );
}

// Save JPG

function saveJPG(filePath){
var jpegOptions = new JPEGSaveOptions();
jpegOptions.quality = 10;
jpegOptions.embedColorProfile = false;
app.activeDocument.saveAs( File( filePath), jpegOptions, false);
}

function resolveColorModes(){
	// Remove all alpha channels
	app.activeDocument.channels.removeAll();

	if(app.activeDocument.bitsPerChannel == BitsPerChannelType.SIXTEEN || app.activeDocument.bitsPerChannel == BitsPerChannelType.THIRTYTWO) {

		app.activeDocument.bitsPerChannel = BitsPerChannelType.EIGHT;

		}
	if(app.activeDocument.mode == DocumentMode.BITMAP) {

	app.activeDocument.changeMode(ChangeMode.GRAYSCALE);
	app.activeDocument.changeMode(ChangeMode.RGB);

	} else if (app.activeDocument.mode == DocumentMode.CMYK){
	app.activeDocument.changeMode(ChangeMode.RGB);
	} else if (app.activeDocument.mode == DocumentMode.INDEXEDCOLOR){
	app.activeDocument.changeMode(ChangeMode.RGB);
	}
}

///////////////////////////////
// Document Set-up Functions
///////////////////////////////

// Write out HTML File

    function writeHTMLDoc(galleryTitle, backgroundColor) {
        writeHTML(
			'<!DOCTYPE html>\n' +
			'<html lang="en">\n' +
			'	<head>\n' +
			'		<title>' + escapeXML(galleryTitle) + '</title>\n' +
			'		<meta charset="utf-8" />\n' +
			'		<meta name="viewport" id="jb-viewport" content="width=device-width, initial-scale=1.0, minimum-scale=1, maximum-scale=1, user-scalable=0" />\n' +
			'		<meta name="description" content="" />\n' +
			'		<!--START OPEN GRAPH TAGS-->\n' +
			'		<meta property="og:description" content="" />\n' +
			'		<meta property="og:image" content="" />\n' +
			'		<meta property="og:title" content="' + escapeDoubleQuotes(galleryTitle) + '" />\n' +
			'		<meta property="og:type" content="website" />\n' +
			'		<meta property="og:url" content="" />\n' +
			'		<!--END OPEN GRAPH TAGS-->\n' +
			'		<style type="text/css">\n' +
			'			body {\n' +
			'				margin: 0px;\n' +
			'			}\n' +
			'		</style>\n' +
			'	</head>\n' +
			'	<body>\n' +
			'		<!--START JUICEBOX EMBED.-->\n' +
			'		<script src="jbcore/juicebox.js"></script>\n' +
			'		<script>\n' +
			'			new juicebox({\n' +
			'				backgroundColor: "' + backgroundColor + '",\n' +
			'				containerId: "juicebox-container",\n' +
			'				galleryHeight: "100%",\n' +
			'				galleryWidth: "100%"\n' +
			'			});\n' +
			'		</script>\n' +
			'		<div id="juicebox-container"></div>\n' +
			'		<!--END JUICEBOX EMBED.-->\n' +
			'	</body>\n' +
			'</html>\n'
		);
    }

// CheckVersion
function CheckVersion() {
	var numberArray = version.split(".");
	if ( numberArray[0] < 9 ) {
		alert( "You must use Photoshop CS2 or later to run this script!" );
		throw( "You must use Photoshop CS2 or later to run this script!" );
	}
}

// load my params from the xml file on disk if it exists
function LoadParamsFromDisk ( loadFile, params ) {
	if ( loadFile.exists ) {
		loadFile.open( "r" );
		var projectSpace = ReadHeader( loadFile );
		if ( projectSpace == GetScriptNameForXML() ) {
			while ( ! loadFile.eof ) {
				var starter = ReadHeader( loadFile );
				var data = ReadData( loadFile );
				var ender = ReadHeader( loadFile );
				if ( ( "/" + starter ) == ender ) {
					params[starter] = data;
				}
				// force boolean values to boolean types
				if ( data == "true" || data == "false" ) {
					params[starter] = data == "true";
				}
			}
		}
		loadFile.close();
		if ( params["version"] != gVersion ) {
			// do something here to fix version conflicts
			params["version"] = gVersion;
		}
	}
	return params;
}

// save out my params
function SaveParamsToDisk ( saveFile, params ) {
	saveFile.encoding = "UTF8";
	saveFile.open( "w", "TEXT", "????" );
	// unicode signature, this is UTF16 but will convert to UTF8 "EF BB BF"
	saveFile.write("\uFEFF");
	var scriptNameForXML = GetScriptNameForXML();
	saveFile.writeln( "<" + scriptNameForXML + ">" );
	for ( var p in params ) {
		saveFile.writeln( "	<" + p + ">" + params[p] + "</" + p + ">" );
	}
	saveFile.writeln( "</" + scriptNameForXML + ">" );
	saveFile.close();
}

// you can't save certain characters in xml, strip them here
// this list is not complete
function GetScriptNameForXML () {
	var scriptNameForXML = new String( gScriptName );
	var charsToStrip = Array( " ", "'", "." );
	for (var a = 0; a < charsToStrip.length; a++ )  {
		var nameArray = scriptNameForXML.split( charsToStrip[a] );
		scriptNameForXML = "";
		for ( var b = 0; b < nameArray.length; b++ ) {
			scriptNameForXML += nameArray[b];
		}
	}
	return scriptNameForXML;
}

// figure out what I call my params file
function GetDefaultParamsFile() {
	var paramsFolder = new Folder( path + "/Presets/" + gScriptName );
	paramsFolder.create();
	return ( new File( paramsFolder + "/" + gScriptName + ".xml" ) );
}

// a very crude xml parser, this reads the "Tag" of the <Tag>Data</Tag>
function ReadHeader( inFile ) {
	var returnValue = "";
	if ( ! inFile.eof ) {
		var c = "";
		while ( c != "<" && ! inFile.eof ) {
			c = inFile.read( 1 );
		}
		while ( c != ">" && ! inFile.eof ) {
			c = inFile.read( 1 );
			if ( c != ">" ) {
				returnValue += c;
			}
		}
	} else {
		returnValue = "end of file";
	}
	return returnValue;
}

// a very crude xml parser, this reads the "Data" of the <Tag>Data</Tag>
function ReadData( inFile ) {
	var returnValue = "";
	if ( ! inFile.eof ) {
		var c = "";
		while ( c != "<" && ! inFile.eof ) {
			c = inFile.read( 1 );
			if ( c != "<" ) {
				returnValue += c;
			}
		}
		inFile.seek( -1, 1 );
	}
	return returnValue;
}

function escapeXML(escapeXMLInput) {
	var escapeXMLOutput = escapeXMLInput.replace(/&(?!(?:amp;|apos;|gt;|lt;|quot;))/g, '&amp;').replace(/'/g, '&apos;').replace(/>/g, '&gt;').replace(/</g, '&lt;').replace(/"/g, '&quot;');
	return escapeXMLOutput;
}

function escapeDoubleQuotes(escapeDoubleQuotesInput) {
	var escapeDoubleQuotesOutput = escapeDoubleQuotesInput.replace(/"/g, '&quot;');
	return escapeDoubleQuotesOutput;
}

function lineBreak(lineBreakInput) {
	var lineBreakOutput = lineBreakInput.replace(/(?:\r\n|\r|\n)/g, '<br />');
	return lineBreakOutput;
}

function stripControlCharacters(stripControlCharactersInput) {
	var stripControlCharactersOutput = stripControlCharactersInput.replace(/[\x00-\x1F\x7F-\x9F]/g, '');
	return stripControlCharactersOutput;
}

function escapeCDATA(escapeCDATAInput) {
	var escapeCDATAOutput = escapeCDATAInput.replace(/]]>/g, ']]]]><![CDATA[>');
	return escapeCDATAOutput;
}

function getRGBA(hexadecimal, opacity) {
	if (hexadecimal.charAt(0) == '#') {
		hexadecimal = hexadecimal.substring(1, 7);
	}
	var getRGBAOutput = 'rgba(' + parseInt(hexadecimal.substring(0, 2), 16).toString() + ',' + parseInt(hexadecimal.substring(2, 4), 16).toString() + ',' + parseInt(hexadecimal.substring(4, 6), 16).toString() + ',' + opacity + ')';
	return getRGBAOutput;
}
